<?php
// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set headers
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

try {
    // Include database configuration
    require_once __DIR__ . '/../config.php';
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate input
    if (empty($input['url'])) {
        throw new Exception('URL is required');
    }
    
    // Sanitize input
    $url = filter_var($input['url'], FILTER_SANITIZE_URL);
    $title = htmlspecialchars($input['title'] ?? '', ENT_QUOTES, 'UTF-8');
    $description = htmlspecialchars($input['description'] ?? '', ENT_QUOTES, 'UTF-8');
    $is_active = isset($input['is_active']) ? (bool)$input['is_active'] : true;
    $thumbnail_url = !empty($input['thumbnail_url']) ? filter_var($input['thumbnail_url'], FILTER_SANITIZE_URL) : null;
    
    // Validate URL
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
        throw new Exception('Invalid URL format');
    }
    
    // Validate thumbnail URL if provided
    if ($thumbnail_url && !filter_var($thumbnail_url, FILTER_VALIDATE_URL)) {
        throw new Exception('Invalid thumbnail URL format');
    }
    
    // Get database connection
    $pdo = getDBConnection();
    
    // Insert new webview item with thumbnail
    $stmt = $pdo->prepare("
        INSERT INTO webview_urls (
            url, 
            title, 
            description, 
            thumbnail_url,
            is_active
        ) VALUES (
            :url, 
            :title, 
            :description, 
            :thumbnail_url,
            :is_active
        )
    ");
    
    $stmt->execute([
        ':url' => $url,
        ':title' => $title ?: 'New Website',
        ':description' => $description,
        ':thumbnail_url' => $thumbnail_url,
        ':is_active' => $is_active ? 1 : 0
    ]);
    
    $id = $pdo->lastInsertId();
    
    // Return success response
    echo json_encode([
        'success' => true,
        'message' => 'WebView item added successfully',
        'id' => $id
    ]);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}